// 快速填充助手 - 侧边栏脚本（与原插件模块结构一致）

let config = {};
let currentInputInfo = null;
let editingFieldId = null;

// 模块定义（与原插件保持一致）
const modules = [
  { title: '个人信息', id: 'module-personalInfo', key: 'personalInfo', type: 'single' },
  { title: '求职意向', id: 'module-education', key: 'educationList', type: 'list' },
  { title: '教育经历', id: 'module-work', key: 'workList', type: 'list' },
  { title: '工作实习', id: 'module-project', key: 'projectList', type: 'list' },
  { title: '项目经历', id: 'module-selfDescription', key: 'selfDescription', type: 'single' },
  { title: '实践活动', id: 'module-practiceActivity', key: 'practiceActivityList', type: 'list' },
  { title: '培训经历', id: 'module-training', key: 'trainingList', type: 'list' },
  { title: '自我描述', id: 'module-award', key: 'awardList', type: 'list' },
  { title: '其他信息', id: 'module-otherInfo', key: 'otherInfo', type: 'single' }
];

// 初始化
async function init() {
  // 加载配置
  await loadConfig();
  
  // 渲染快捷导航
  renderNavigation();
  
  // 渲染所有模块
  renderAllModules();
  
  // 绑定事件
  bindEvents();
  
  // 监听来自content script的消息
  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'inputFocused') {
      updateTargetInput(request.inputInfo);
    }
    return true;
  });
  
  // 定期检查当前输入框
  setInterval(checkActiveInput, 1000);
}

// 加载配置
async function loadConfig() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['config'], (result) => {
      if (result.config) {
        config = result.config;
        // 确保所有必需的模块都存在，如果缺失则使用默认值
        const defaultConfig = getDefaultConfig();
        modules.forEach(module => {
          if (!config[module.key]) {
            config[module.key] = defaultConfig[module.key];
          }
        });
        // 确保fieldOrder存在
        if (!config.fieldOrder) {
          config.fieldOrder = defaultConfig.fieldOrder;
        }
        // 确保fieldTypes存在（如果getDefaultConfig中有的话）
        if (!config.fieldTypes && defaultConfig.fieldTypes) {
          config.fieldTypes = defaultConfig.fieldTypes;
        }
        // 保存修复后的配置
        saveConfig();
      } else {
        // 如果没有配置，使用默认配置
        config = getDefaultConfig();
        saveConfig();
      }
      console.log('配置已加载:', config);
      resolve();
    });
  });
}

// 获取默认配置
function getDefaultConfig() {
  return {
    "personalInfo": {
      "姓名": "陈晓明",
      "性别": "男",
      "出生日期": "1998-05-15",
      "身份证号": "320102199805150012",
      "籍贯": "江苏省苏州市",
      "现居住地": "江苏省苏州市",
      "手机号码": "13812345678",
      "邮箱": "chenxiaoming@qq.com",
      "微信号": "wxid_ycn0z877kgaq22",
      "紧急联系人": "陈建华",
      "紧急联系电话": "13987654321",
      "政治面貌": "中共党员",
      "最高学历": "本科"
    },
    "educationList": [
      {
        "id": "edu_1",
        "期望工作城市": "上海市",
        "意向岗位": "前端开发工程师",
        "期望薪资": "12000-18000元/月",
        "预计入职时间": "2024-07-01"
      }
    ],
    "workList": [
      {
        "id": "work_1",
        "开始时间": "2020-09-01",
        "结束时间": "2024-06-30",
        "学校名称": "苏州大学",
        "专业名称": "计算机科学与技术",
        "学历": "本科",
        "学习形式": "全国普通高等院校全日制"
      }
    ],
    "projectList": [
      {
        "id": "project_1",
        "开始时间": "2023-07-01",
        "结束时间": "2023-12-31",
        "公司名称": "阿里巴巴集团",
        "职位名称": "前端开发实习生",
        "工作内容": "负责电商平台前端页面开发，使用React框架开发用户界面，参与产品功能迭代和优化",
        "工作成果": "完成了5个核心功能模块的前端开发，提升了用户体验，获得团队一致好评"
      }
    ],
    "selfDescription": {
      "开始时间": "2022-09-01",
      "结束时间": "2023-05-31",
      "项目名称": "在线学习平台",
      "职位名称": "前端开发负责人",
      "本人职责": "负责整个平台的前端架构设计和开发，使用React+TypeScript技术栈，协调UI设计师和后端开发团队",
      "项目成果": "成功上线了功能完整的在线学习平台，用户注册量超过5000人，获得了学校优秀项目奖"
    },
    "practiceActivityList": [
      {
        "id": "practice_1",
        "开始时间": "2023-07-01",
        "结束时间": "2023-09-30",
        "实践名称": "智慧校园社会实践项目",
        "实践描述": "参与校园信息化建设实践，负责用户需求调研和系统功能测试，协助完成移动端应用开发，提升了用户体验满意度20%"
      }
    ],
    "trainingList": [
      {
        "id": "training_1",
        "开始时间": "2024-01-10",
        "结束时间": "2024-02-10",
        "培训名称": "Vue.js企业级应用开发实战",
        "培训机构": "阿里巴巴技术学院",
        "培训描述": "系统学习Vue3组合式API、状态管理、路由配置等企业级开发技能，完成电商管理系统实战项目，掌握前后端分离架构设计"
      }
    ],
    "awardList": [
      {
        "id": "award_1",
        "自我评价": "我是一个技术热情、学习能力强的前端开发者，具备扎实的编程基础和良好的团队协作能力。热爱新技术，善于解决问题，能够在快节奏的工作环境中保持高效产出。"
      }
    ],
    "otherInfo": {
      "家庭情况": "家庭和睦，父母都是普通工人，从小培养了我勤劳朴实、踏实肯干的品质，注重实用技能的学习",
      "著作专利": "参与开发开源项目2个，获得软件著作权1项，在技术博客发表文章10余篇",
      "干部任职情况": "担任计算机学院学生会技术部副部长，组织过3次技术分享会，参与过多个开源项目",
      "证书名称": "全国计算机等级考试三级网络技术证书"
    },
    "fieldOrder": {
      "personalInfo": ["姓名", "性别", "出生日期", "身份证号", "籍贯", "现居住地", "手机号码", "邮箱", "微信号", "紧急联系人", "紧急联系电话", "政治面貌", "最高学历"],
      "selfDescription": ["开始时间", "结束时间", "项目名称", "职位名称", "本人职责", "项目成果"],
      "otherInfo": ["家庭情况", "著作专利", "干部任职情况", "证书名称"],
      "education": ["期望工作城市", "意向岗位", "期望薪资", "预计入职时间"],
      "work": ["开始时间", "结束时间", "学校名称", "专业名称", "学历", "学习形式"],
      "project": ["开始时间", "结束时间", "公司名称", "职位名称", "工作内容", "工作成果"],
      "practiceActivity": ["开始时间", "结束时间", "实践名称", "实践描述"],
      "training": ["开始时间", "结束时间", "培训名称", "培训机构", "培训描述"],
      "award": ["自我评价"]
    }
  };
}

// 保存配置
async function saveConfig() {
  return new Promise((resolve) => {
    chrome.storage.local.set({ 'config': config }, () => {
      resolve();
    });
  });
}

// 渲染快捷导航
function renderNavigation() {
  const navContainer = document.getElementById('moduleNav');
  if (!navContainer) return;
  
  navContainer.innerHTML = '';
  
  modules.forEach(module => {
    const navBtn = document.createElement('button');
    navBtn.className = 'module-nav-btn';
    navBtn.textContent = module.title;
    navBtn.onclick = () => {
      scrollToModule(module.id);
    };
    navContainer.appendChild(navBtn);
  });
}

// 滚动到指定模块
function scrollToModule(moduleId) {
  const module = document.getElementById(moduleId);
  if (module) {
    // 获取快捷导航区的位置和高度
    const navWrapper = document.querySelector('.module-nav-wrapper');
    let offset = 0;
    
    if (navWrapper) {
      const navRect = navWrapper.getBoundingClientRect();
      const container = document.querySelector('.container');
      const containerRect = container.getBoundingClientRect();
      
      // 计算快捷导航区底部相对于容器的位置
      const navBottom = navRect.bottom - containerRect.top;
      // 目标位置：快捷导航区底部 + 7mm（26.46px，即1cm减去3mm）
      offset = navBottom + 26.46;
    }
    
    // 计算模块相对于容器的位置
    const container = document.querySelector('.container');
    const moduleRect = module.getBoundingClientRect();
    const containerRect = container.getBoundingClientRect();
    const moduleTop = moduleRect.top - containerRect.top;
    
    // 计算需要滚动的距离：模块当前位置 - 目标位置
    const scrollOffset = moduleTop - offset;
    
    // 平滑滚动
    container.scrollTo({
      top: container.scrollTop + scrollOffset,
      behavior: 'smooth'
    });
    
    // 添加高亮效果
    module.style.boxShadow = '0 0 20px rgba(76, 175, 80, 0.5)';
    setTimeout(() => {
      module.style.boxShadow = '';
    }, 2000);
  }
}

// 渲染所有模块
function renderAllModules() {
  const container = document.getElementById('fieldsContainer');
  if (!container) return;
  
  container.innerHTML = '';
  
  modules.forEach(module => {
    const moduleElement = createModule(module);
    if (moduleElement) {
      container.appendChild(moduleElement);
    }
  });
}

// 创建模块
function createModule(moduleInfo) {
  const moduleDiv = document.createElement('div');
  moduleDiv.id = moduleInfo.id;
  moduleDiv.className = 'fill-module';
  moduleDiv.style.cssText = 'margin-bottom: 16px;';
  
  // 模块标题
  const moduleHeader = document.createElement('div');
  moduleHeader.className = 'module-header';
  
  const moduleTitle = document.createElement('h5');
  moduleTitle.textContent = moduleInfo.title;
  
  // 收起/展开按钮（放在标题右侧）
  const collapseBtn = document.createElement('button');
  collapseBtn.className = 'collapse-btn';
  collapseBtn.textContent = '▼';
  collapseBtn.onclick = () => toggleModule(collapseBtn);
  
  // 添加字段按钮（放在最右侧）
  const addFieldBtn = document.createElement('button');
  addFieldBtn.className = 'add-field-btn';
  addFieldBtn.innerHTML = '<span class="add-field-icon">+</span><span class="add-field-text">添加字段</span>';
  addFieldBtn.onclick = (e) => {
    e.stopPropagation();
    addFieldToModule(moduleInfo);
  };
  addFieldBtn.style.marginLeft = 'auto';
  addFieldBtn.style.marginRight = '0';
  
  // 按顺序添加：标题 -> 展开/收起按钮 -> 添加字段按钮
  moduleHeader.appendChild(moduleTitle);
  moduleHeader.appendChild(collapseBtn);
  moduleHeader.appendChild(addFieldBtn);
  moduleDiv.appendChild(moduleHeader);
  
  // 模块内容
  const moduleContent = document.createElement('div');
  moduleContent.className = 'module-content';
  
  const fillButtons = document.createElement('div');
  fillButtons.className = 'fill-buttons';
  
  // 根据模块类型渲染字段
  if (moduleInfo.type === 'single') {
    renderSingleModuleFields(moduleInfo, fillButtons);
  } else {
    renderListModuleFields(moduleInfo, fillButtons);
  }
  
  moduleContent.appendChild(fillButtons);
  moduleDiv.appendChild(moduleContent);
  
  return moduleDiv;
}

// 模块展开/收起功能
function toggleModule(button) {
  const moduleContent = button.closest('.fill-module').querySelector('.module-content');
  const isCollapsed = moduleContent.classList.contains('collapsed');
  
  if (isCollapsed) {
    moduleContent.classList.remove('collapsed');
    button.textContent = '▼';
  } else {
    moduleContent.classList.add('collapsed');
    button.textContent = '▶';
  }
}

// 渲染单组模块字段
function renderSingleModuleFields(moduleInfo, container) {
  let moduleData = config[moduleInfo.key];
  
  // 如果数据不存在或为空，使用默认数据
  if (!moduleData || typeof moduleData !== 'object' || Object.keys(moduleData).length === 0) {
    // 尝试从默认配置获取
    const defaultConfig = getDefaultConfig();
    moduleData = defaultConfig[moduleInfo.key] || {};
    
    // 更新config
    config[moduleInfo.key] = moduleData;
  }
  
  const fieldOrder = (config.fieldOrder && config.fieldOrder[moduleInfo.key]) || Object.keys(moduleData);
  
  // 如果fieldOrder为空，使用默认配置的fieldOrder
  if (!fieldOrder || fieldOrder.length === 0) {
    const defaultConfig = getDefaultConfig();
    const defaultFieldOrder = defaultConfig.fieldOrder && defaultConfig.fieldOrder[moduleInfo.key];
    if (defaultFieldOrder && defaultFieldOrder.length > 0) {
      defaultFieldOrder.forEach(fieldName => {
        const fieldValue = moduleData[fieldName] || '';
        const fieldElement = createFieldElement(moduleInfo.key, fieldName, fieldValue, moduleInfo.id);
        container.appendChild(fieldElement);
      });
      return;
    }
  }
  
  fieldOrder.forEach(fieldName => {
    const fieldValue = moduleData[fieldName] || '';
    const fieldElement = createFieldElement(moduleInfo.key, fieldName, fieldValue, moduleInfo.id);
    container.appendChild(fieldElement);
  });
}

// 渲染列表模块字段
function renderListModuleFields(moduleInfo, container) {
  let moduleData = config[moduleInfo.key];
  
  // 如果数据不存在或为空，使用默认数据
  if (!moduleData || !Array.isArray(moduleData) || moduleData.length === 0) {
    // 尝试从默认配置获取
    const defaultConfig = getDefaultConfig();
    moduleData = defaultConfig[moduleInfo.key] || [];
    
    // 如果默认配置也没有，创建一个空项
    if (!Array.isArray(moduleData) || moduleData.length === 0) {
      moduleData = [{}];
    }
    
    // 更新config
    config[moduleInfo.key] = moduleData;
  }
  
  // 只显示第一项的字段
  const firstItem = moduleData[0] || {};
  const moduleKey = moduleInfo.key.replace('List', '');
  const fieldOrder = (config.fieldOrder && config.fieldOrder[moduleKey]) || Object.keys(firstItem).filter(k => k !== 'id');
  
  // 如果fieldOrder为空，使用默认配置的fieldOrder
  if (!fieldOrder || fieldOrder.length === 0) {
    const defaultConfig = getDefaultConfig();
    const defaultFieldOrder = defaultConfig.fieldOrder && defaultConfig.fieldOrder[moduleKey];
    if (defaultFieldOrder && defaultFieldOrder.length > 0) {
      defaultFieldOrder.forEach(fieldName => {
        const fieldValue = firstItem[fieldName] || '';
        const fieldElement = createFieldElement(moduleInfo.key, fieldName, fieldValue, moduleInfo.id, 0);
        container.appendChild(fieldElement);
      });
      return;
    }
  }
  
  fieldOrder.forEach(fieldName => {
    const fieldValue = firstItem[fieldName] || '';
    const fieldElement = createFieldElement(moduleInfo.key, fieldName, fieldValue, moduleInfo.id, 0);
    container.appendChild(fieldElement);
  });
}

// 创建字段元素
function createFieldElement(moduleKey, fieldName, fieldValue, moduleId, itemIndex = null) {
  const fillBtn = document.createElement('div');
  fillBtn.className = 'fill-btn';
  const fieldId = itemIndex !== null ? `field_${moduleKey}_${itemIndex}_${fieldName}` : `field_${moduleKey}_${fieldName}`;
  fillBtn.id = fieldId;
  fillBtn.setAttribute('data-module-key', moduleKey);
  fillBtn.setAttribute('data-field-name', fieldName);
  if (itemIndex !== null) {
    fillBtn.setAttribute('data-item-index', itemIndex);
  }
  
  // 删除按钮（在字段按钮右上角）
  const deleteBtn = document.createElement('button');
  deleteBtn.className = 'field-delete-btn';
  deleteBtn.setAttribute('data-field', fieldName);
  deleteBtn.innerHTML = '<span class="field-delete-icon">×</span>';
  deleteBtn.onclick = (e) => {
    e.stopPropagation();
    e.preventDefault();
    deleteField(moduleKey, fieldName, itemIndex);
  };
  
  // 字段内容容器
  const fieldContent = document.createElement('div');
  fieldContent.className = 'field-content';
  fieldContent.style.cssText = 'width: 100%; padding: 8px; position: relative; display: flex; flex-direction: row; align-items: flex-start; overflow: visible; min-height: 30px;';
  
  // 粘贴按钮（左侧）
  const fillBtnIcon = document.createElement('button');
  fillBtnIcon.className = 'field-fill-btn';
  fillBtnIcon.setAttribute('data-field', fieldName);
  fillBtnIcon.innerHTML = '<span style="color: white !important; font-size: 18px !important; font-weight: bold !important;">📋</span>';
  fillBtnIcon.title = '点击填充到网页输入框';
  fillBtnIcon.onclick = (e) => {
    e.preventDefault();
    e.stopPropagation();
    e.stopImmediatePropagation();
    fillInput(fieldValue);
  };
  
  // 字段名称
  const fieldNameSpan = document.createElement('div');
  fieldNameSpan.className = 'field-name';
  fieldNameSpan.textContent = fieldName + '：';
  fieldNameSpan.style.cssText = 'font-weight: bold; color: #4285f4; display: inline-block; line-height: 1.4; font-size: 12px !important; white-space: nowrap; flex-shrink: 0; position: absolute; left: 36px; top: 50%; transform: translateY(-50%); z-index: 2; background: transparent; pointer-events: none;';
  
  // 在侧边栏模式下，字段名称颜色会被CSS覆盖为绿色
  
  // 字段值容器（可编辑）
  const valueContainer = document.createElement('div');
  valueContainer.className = 'field-value-container editable-field';
  valueContainer.setAttribute('data-field', fieldName);
  
  // 计算字段名称宽度（动态计算）
  let marginLeft = 36 + (fieldName + '：').length * 6; // 基础宽度 + 字段名宽度
  try {
    // 创建临时元素来测量字段名称的实际宽度
    const tempElement = document.createElement('span');
    tempElement.style.position = 'absolute';
    tempElement.style.visibility = 'hidden';
    tempElement.style.whiteSpace = 'nowrap';
    tempElement.style.fontSize = '12px';
    tempElement.style.fontFamily = 'Arial, sans-serif';
    tempElement.style.fontWeight = 'bold';
    tempElement.textContent = fieldName + '：';
    document.body.appendChild(tempElement);
    const fieldNameWidth = tempElement.offsetWidth;
    document.body.removeChild(tempElement);
    marginLeft = fieldNameWidth + 28; // 字段名宽度 + 粘贴按钮宽度(20px) + 间距(8px)
  } catch (error) {
    console.error('字段名称宽度测量失败:', error);
  }
  
  valueContainer.style.cssText = `flex: 1; min-height: 24px; padding: 6px 8px; font-size: 12px; color: #333; text-align: left; cursor: pointer; transition: all 0.2s ease; display: flex; align-items: flex-start; word-wrap: break-word; white-space: pre-wrap; overflow-wrap: break-word; word-break: break-all; line-height: 1.4; position: relative; margin-left: ${marginLeft}px; overflow-x: hidden;`;
  
  // 字段值文本显示
  const valueText = document.createElement('span');
  valueText.className = 'field-value-text';
  valueText.textContent = fieldValue || '未设置';
  valueText.style.cssText = 'word-wrap: break-word; white-space: pre-wrap; overflow-wrap: break-word; word-break: break-all; overflow-x: hidden; width: 100%;';
  
  // 判断是否使用textarea（与原插件逻辑一致）
  const fieldType = getFieldType(moduleKey, fieldName);
  const forceTextareaFields = ['项目成果', '家庭情况', '著作专利', '工作内容', '工作成果', '干部任职情况', '本人职责', '项目成果', '实践描述', '培训描述', '自我评价'];
  let shouldUseTextarea = false;
  
  if (forceTextareaFields.includes(fieldName)) {
    shouldUseTextarea = true;
  } else if (fieldType === 'textarea') {
    shouldUseTextarea = true;
  } else {
    // 检查是否包含换行符
    if (fieldValue && (fieldValue.includes('\n') || fieldValue.includes('\r'))) {
      shouldUseTextarea = true;
    } else {
      // 检查文本长度
      if (fieldValue && fieldValue.length > 30) {
        shouldUseTextarea = true;
      }
    }
  }
  
  // 字段值输入框（编辑模式）
  let valueInput;
  if (fieldType === 'date') {
    // 日期类型使用date input
    valueInput = document.createElement('input');
    valueInput.type = 'date';
    valueInput.className = 'field-value-input';
    valueInput.value = fieldValue;
    valueInput.style.display = 'none';
    valueInput.style.cssText = 'display: none; width: 100%; padding: 8px 12px; border: 2px solid #007bff; border-radius: 8px; font-size: 11px; background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%); box-sizing: border-box; box-shadow: 0 2px 8px rgba(0, 123, 255, 0.15); transition: all 0.3s ease; outline: none; font-family: inherit;';
  } else if (shouldUseTextarea) {
    // 多行文本使用textarea，启用垂直滚动条
    valueInput = document.createElement('textarea');
    valueInput.className = 'field-value-input';
    valueInput.value = fieldValue;
    valueInput.style.display = 'none';
    valueInput.placeholder = '请输入字段值';
    valueInput.style.cssText = 'display: none; width: 100%; padding: 8px 12px; border: 2px solid #007bff; border-radius: 8px; font-size: 11px; background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%); box-sizing: border-box; box-shadow: 0 2px 8px rgba(0, 123, 255, 0.15); transition: all 0.3s ease; outline: none; font-family: inherit; resize: vertical; min-height: 60px; overflow-y: auto;';
  } else {
    // 单行文本使用input
    valueInput = document.createElement('input');
  valueInput.type = 'text';
  valueInput.className = 'field-value-input';
  valueInput.value = fieldValue;
  valueInput.style.display = 'none';
  valueInput.placeholder = '请输入字段值';
    valueInput.style.cssText = 'display: none; width: 100%; padding: 8px 12px; border: 2px solid #007bff; border-radius: 8px; font-size: 11px; background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%); box-sizing: border-box; box-shadow: 0 2px 8px rgba(0, 123, 255, 0.15); transition: all 0.3s ease; outline: none; font-family: inherit;';
  }
  
  // 组装字段值容器
  valueContainer.appendChild(valueText);
  valueContainer.appendChild(valueInput);
  
  // 组装字段内容
  fieldContent.appendChild(fillBtnIcon);
  fieldContent.appendChild(fieldNameSpan);
  fieldContent.appendChild(valueContainer);
  
  // 组装字段按钮
  fillBtn.appendChild(deleteBtn);
  fillBtn.appendChild(fieldContent);
  
  // 字段值容器双击事件（显示输入框）
  valueContainer.addEventListener('dblclick', () => {
    valueText.style.display = 'none';
    valueInput.style.display = 'block';
    valueInput.style.border = '2px solid #007bff';
    valueInput.style.boxShadow = '0 4px 12px rgba(0, 123, 255, 0.25)';
    valueInput.style.transform = 'scale(1.02)';
    
    // 如果是textarea，确保启用垂直滚动条（与原插件一致）
    if (valueInput.tagName === 'TEXTAREA') {
      valueInput.style.overflowY = 'auto';
      valueInput.style.resize = 'vertical';
      valueInput.style.minHeight = '60px';
      // 根据内容高度设置最大高度，超过一定高度时启用滚动条
      const lines = calculateTextLines(fieldValue || '未设置', valueContainer.offsetWidth || 200, 12, 1.4);
      if (lines > 5) {
        valueInput.style.maxHeight = '150px';
      } else {
        valueInput.style.maxHeight = 'none';
      }
    }
    
    valueInput.focus();
    if (valueInput.type !== 'date' && valueInput.tagName !== 'TEXTAREA') {
      valueInput.select();
    }
  });
  
  // 输入框事件
  valueInput.onkeydown = (e) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      saveFieldValue(fieldId, moduleKey, fieldName, valueInput.value, itemIndex);
      valueInput.blur();
    } else if (e.key === 'Escape') {
      cancelEditField(fieldId, moduleKey, fieldName, itemIndex);
      valueInput.blur();
    }
  };
  
  valueInput.onblur = () => {
    saveFieldValue(fieldId, moduleKey, fieldName, valueInput.value, itemIndex);
    valueText.style.display = 'block';
    valueInput.style.display = 'none';
    valueInput.style.transform = 'scale(1)';
  };
  
  // 粘贴按钮悬停效果
  fillBtnIcon.addEventListener('mouseenter', () => {
    fillBtnIcon.style.transform = 'translateY(-50%) scale(1.1)';
    fillBtnIcon.style.boxShadow = '0 4px 8px rgba(76, 175, 80, 0.5)';
  });
  
  fillBtnIcon.addEventListener('mouseleave', () => {
    fillBtnIcon.style.transform = 'translateY(-50%) scale(1)';
    fillBtnIcon.style.boxShadow = '0 2px 4px rgba(76, 175, 80, 0.3)';
  });
  
  // 根据文本行数调整按钮高度和滚动条
  setTimeout(() => {
    try {
      const containerWidth = valueContainer.offsetWidth || 200;
      const lines = calculateTextLines(fieldValue || '未设置', containerWidth, 12, 1.4);
      adjustButtonHeight(fillBtn, fieldValue || '未设置', containerWidth);
      
      // 设置粘贴按钮的宽度
      // 宽度使用固定值（基于单行字段的基准高度40px减去2mm），保持所有字段一致
      // 高度通过CSS的100%自动适应字段按钮高度，无需手动设置
      const baseWidth = 40 - 7.56; // 基准宽度：40px（单行字段高度）减去2mm（7.56px）= 32.44px
      fillBtnIcon.style.setProperty('width', baseWidth + 'px', 'important');
      
      // 根据行数设置滚动条（与原插件保持一致）
      // 单行文本：无垂直滚动条
      if (lines <= 1) {
        valueContainer.style.overflowY = 'visible';
        valueContainer.style.maxHeight = 'none';
        valueText.style.overflowY = 'visible';
        valueText.style.maxHeight = 'none';
      } else {
        // 多行文本：启用垂直滚动条
        // 计算实际高度（基础高度30px + 每行13px）
        const actualHeight = 30 + (lines - 1) * 13;
        // 如果高度超过120px，启用滚动条并限制最大高度
        if (actualHeight > 120) {
          valueContainer.style.overflowY = 'auto';
          valueContainer.style.maxHeight = '120px'; // 限制最大显示高度
          valueText.style.overflowY = 'auto';
          valueText.style.maxHeight = '120px';
        } else {
          // 内容不多，不启用滚动条，让按钮高度自适应
          valueContainer.style.overflowY = 'visible';
          valueContainer.style.maxHeight = 'none';
          valueText.style.overflowY = 'visible';
          valueText.style.maxHeight = 'none';
        }
      }
    } catch (error) {
      console.error('调整按钮高度时出错:', error);
    }
  }, 100);
  
  return fillBtn;
}

// 切换编辑模式（已改为悬停编辑，此函数保留但不使用）
function toggleEditField(fieldId, moduleKey, fieldName, itemIndex) {
  // 编辑功能已改为悬停时自动显示输入框
}

// 保存字段值
async function saveFieldValue(fieldId, moduleKey, fieldName, value, itemIndex) {
  if (itemIndex !== null) {
    // 列表类型
    if (!config[moduleKey]) {
      config[moduleKey] = [];
    }
    if (!config[moduleKey][itemIndex]) {
      config[moduleKey][itemIndex] = { id: `${moduleKey.replace('List', '')}_${Date.now()}` };
    }
    config[moduleKey][itemIndex][fieldName] = value;
  } else {
    // 单组类型
    if (!config[moduleKey]) {
      config[moduleKey] = {};
    }
    config[moduleKey][fieldName] = value;
  }
  
  await saveConfig();
  
  const field = document.getElementById(fieldId);
  if (field) {
    const valueText = field.querySelector('.field-value-text');
    const valueInput = field.querySelector('.field-value-input');
    const valueContainer = field.querySelector('.field-value-container');
    valueText.style.display = 'block';
    valueInput.style.display = 'none';
    valueText.textContent = value || '未设置';
    valueInput.value = value;
    valueInput.style.transform = 'scale(1)';
    
    // 重新调整按钮高度和滚动条
    setTimeout(() => {
      try {
        const containerWidth = valueContainer ? valueContainer.offsetWidth || 200 : 200;
        const lines = calculateTextLines(value || '未设置', containerWidth, 12, 1.4);
        adjustButtonHeight(field, value || '未设置', containerWidth);
        
        // 设置粘贴按钮的宽度
        // 宽度使用固定值（基于单行字段的基准高度40px减去2mm），保持所有字段一致
        // 高度通过CSS的100%自动适应字段按钮高度，无需手动设置
        const fillBtnIcon = field.querySelector('.field-fill-btn');
        if (fillBtnIcon) {
          const baseWidth = 40 - 7.56; // 基准宽度：40px（单行字段高度）减去2mm（7.56px）= 32.44px
          fillBtnIcon.style.setProperty('width', baseWidth + 'px', 'important');
        }
        
        // 根据行数设置滚动条
        if (valueContainer) {
          if (lines <= 1) {
            // 单行文本：无垂直滚动条
            valueContainer.style.overflowY = 'visible';
            valueContainer.style.maxHeight = 'none';
            valueText.style.overflowY = 'visible';
            valueText.style.maxHeight = 'none';
          } else {
            // 多行文本：启用垂直滚动条
            const actualHeight = 30 + (lines - 1) * 13;
            if (actualHeight > 120) {
              valueContainer.style.overflowY = 'auto';
              valueContainer.style.maxHeight = '120px';
              valueText.style.overflowY = 'auto';
              valueText.style.maxHeight = '120px';
            } else {
              valueContainer.style.overflowY = 'visible';
              valueContainer.style.maxHeight = 'none';
              valueText.style.overflowY = 'visible';
              valueText.style.maxHeight = 'none';
            }
          }
        }
      } catch (error) {
        console.error('调整按钮高度时出错:', error);
      }
    }, 100);
  }
  
  editingFieldId = null;
  // 已删除提示消息
}

// 取消编辑
function cancelEditField(fieldId, moduleKey, fieldName, itemIndex) {
  const field = document.getElementById(fieldId);
  if (field) {
    const valueText = field.querySelector('.field-value-text');
    const valueInput = field.querySelector('.field-value-input');
    
    let currentValue = '';
    if (itemIndex !== null) {
      const moduleData = config[moduleKey];
      if (Array.isArray(moduleData) && moduleData[itemIndex]) {
        currentValue = moduleData[itemIndex][fieldName] || '';
      }
    } else {
      currentValue = config[moduleKey] && config[moduleKey][fieldName] || '';
    }
    
    valueText.style.display = 'block';
    valueInput.style.display = 'none';
    valueInput.value = currentValue;
    valueText.textContent = currentValue || '未设置';
    valueInput.style.transform = 'scale(1)';
  }
  
  editingFieldId = null;
}

// 删除字段
async function deleteField(moduleKey, fieldName, itemIndex) {
  if (!confirm(`确定要删除字段"${fieldName}"吗？`)) {
    return;
  }
  
  if (itemIndex !== null) {
    // 列表类型：从所有项中删除该字段
    if (config[moduleKey] && Array.isArray(config[moduleKey])) {
      config[moduleKey].forEach(item => {
        if (item.hasOwnProperty(fieldName)) {
          delete item[fieldName];
        }
      });
    }
  } else {
    // 单组类型：直接删除字段
    if (config[moduleKey] && config[moduleKey].hasOwnProperty(fieldName)) {
      delete config[moduleKey][fieldName];
    }
  }
  
  // 从字段顺序中移除
  const orderKey = itemIndex !== null ? moduleKey.replace('List', '') : moduleKey;
  if (config.fieldOrder && config.fieldOrder[orderKey]) {
    config.fieldOrder[orderKey] = config.fieldOrder[orderKey].filter(name => name !== fieldName);
  }
  
  await saveConfig();
  renderAllModules();
  showMessage('字段已删除', 'success');
}

// 添加字段到模块
async function addFieldToModule(moduleInfo) {
  showAddFieldDialog(moduleInfo);
}

// 显示添加字段对话框
function showAddFieldDialog(moduleInfo) {
  // 创建对话框
  const dialog = document.createElement('div');
  dialog.className = 'add-field-dialog';
  dialog.innerHTML = `
    <div class="dialog-overlay"></div>
    <div class="dialog-content">
      <div class="dialog-header">
        <h3>添加新字段</h3>
        <span class="dialog-close">×</span>
      </div>
      <div class="dialog-body" style="padding: 24px; background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);">
        <div class="form-group" style="margin-bottom: 20px;">
          <label style="display: block; font-size: 14px; font-weight: 600; color: #333; margin-bottom: 8px;">
            <span style="color: #4caf50;">✏️</span> 字段名称
          </label>
          <input type="text" id="newFieldName" placeholder="请输入字段名称（最多8个字）" maxlength="8" style="width: 100%; padding: 12px 16px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; font-family: inherit; box-sizing: border-box; transition: all 0.3s ease; background: #ffffff;">
          <div class="field-name-hint" style="font-size: 12px; color: #4caf50; margin-top: 6px; text-align: right; font-weight: 500;">
            字段名称≤8个字
          </div>
        </div>
        <div class="form-group" style="margin-bottom: 20px;">
          <label style="display: block; font-size: 14px; font-weight: 600; color: #333; margin-bottom: 8px;">
            <span style="color: #4caf50;">📋</span> 字段类型
          </label>
          <select id="newFieldType" class="field-type-select" style="width: 100%; padding: 12px 16px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; font-family: inherit; color: #666; background: #ffffff; cursor: pointer; box-sizing: border-box; transition: all 0.3s ease; appearance: none; background-image: url('data:image/svg+xml;charset=UTF-8,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 24 24%22 fill=%22%23666%22><path d=%22M7 10l5 5 5-5z%22/></svg>'); background-repeat: no-repeat; background-position: right 12px center; background-size: 16px;">
            <option value="" class="placeholder-option" selected disabled style="color: #999;">请选择字段类型</option>
            <option value="text" style="color: #333;">📝 文本</option>
            <option value="date" style="color: #333;">📅 日期</option>
          </select>
        </div>
        <div class="form-group" id="fieldValueGroup" style="margin-bottom: 16px;">
          <label style="display: block; font-size: 14px; font-weight: 600; color: #333; margin-bottom: 8px;">
            <span style="color: #4caf50;">💬</span> 字段值
          </label>
          <input type="text" id="newFieldValueDefault" placeholder="请输入字段值" style="width: 100%; padding: 12px 16px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; font-family: inherit; display: block; color: #666; box-sizing: border-box; transition: all 0.3s ease; background: #ffffff;">
          <textarea id="newFieldValueText" placeholder="请输入字段值，支持多行输入" style="width: 100%; padding: 12px 16px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; resize: vertical; min-height: 80px; font-family: inherit; display: none; box-sizing: border-box; transition: all 0.3s ease; background: #ffffff; line-height: 1.6;"></textarea>
          <div id="newFieldValueDateWrapper" style="display: none; position: relative;">
            <input type="text" id="newFieldValueDate" placeholder="请选择日期" readonly style="width: 100%; padding: 12px 40px 12px 16px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 14px; font-family: inherit; color: #666; background: linear-gradient(135deg, #f8f9fa 0%, #f0f0f0 100%); cursor: pointer; box-sizing: border-box; transition: all 0.3s ease;">
            <span id="newFieldValueDateIcon" style="position: absolute; right: 14px; top: 50%; transform: translateY(-50%); color: #4caf50; font-size: 18px; cursor: pointer; pointer-events: none; transition: transform 0.3s ease;">📅</span>
            <input type="date" id="newFieldValueDateHidden" style="position: absolute; opacity: 0; pointer-events: none; width: 0; height: 0;">
          </div>
          <div class="field-value-hint" id="fieldValueHint" style="font-size: 12px; color: #4caf50; margin-top: 6px; text-align: right; display: none; font-weight: 500;">
            ✨ 字段值不限字数
          </div>
        </div>
      </div>
      <div class="dialog-footer">
        <button class="btn btn-cancel" id="cancelAddField">取消</button>
        <button class="btn btn-confirm" id="confirmAddField">确认添加</button>
      </div>
    </div>
  `;
  
  document.body.appendChild(dialog);
  
  // 添加CSS样式
  const style = document.createElement('style');
  style.textContent = `
    .field-type-select option.placeholder-option {
      display: none;
    }
    #newFieldName:hover, #newFieldValueDefault:hover, #newFieldValueText:hover, #newFieldValueDate:hover {
      border-color: #4caf50 !important;
      box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
    }
    #newFieldName:focus, #newFieldValueDefault:focus, #newFieldValueText:focus, #newFieldValueDate:focus {
      border-color: #4caf50 !important;
      box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.15) !important;
      outline: none;
    }
    #newFieldType:hover {
      border-color: #4caf50 !important;
      box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
    }
    #newFieldType:focus {
      border-color: #4caf50 !important;
      box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.15);
      outline: none;
    }
    #newFieldValueDateWrapper:hover #newFieldValueDateIcon {
      transform: translateY(-50%) scale(1.1);
    }
  `;
  document.head.appendChild(style);
  
  // 绑定事件
  const closeBtn = dialog.querySelector('.dialog-close');
  const cancelBtn = dialog.querySelector('#cancelAddField');
  const confirmBtn = dialog.querySelector('#confirmAddField');
  const fieldNameInput = dialog.querySelector('#newFieldName');
  const fieldValueDefault = dialog.querySelector('#newFieldValueDefault');
  const fieldValueText = dialog.querySelector('#newFieldValueText');
  const fieldValueDateWrapper = dialog.querySelector('#newFieldValueDateWrapper');
  const fieldValueDate = dialog.querySelector('#newFieldValueDate');
  const fieldValueDateHidden = dialog.querySelector('#newFieldValueDateHidden');
  const fieldValueHint = dialog.querySelector('#fieldValueHint');
  const fieldTypeSelect = dialog.querySelector('#newFieldType');
  
  // 字段类型选择器颜色控制和显示逻辑
  const updateFieldValueDisplay = () => {
    const selectedValue = fieldTypeSelect.value;
    
    if (selectedValue) {
      fieldTypeSelect.style.color = '#333';
    } else {
      fieldTypeSelect.style.color = '#666';
    }
    
    if (!selectedValue) {
      // 默认状态：显示单行输入框，隐藏提示
      fieldValueDefault.style.display = 'block';
      fieldValueText.style.display = 'none';
      fieldValueDateWrapper.style.display = 'none';
      fieldValueHint.style.display = 'none';
    } else if (selectedValue === 'date') {
      // 日期类型：显示日期输入框，隐藏其他
      fieldValueDefault.style.display = 'none';
      fieldValueText.style.display = 'none';
      fieldValueDateWrapper.style.display = 'block';
      fieldValueHint.style.display = 'none';
    } else {
      // 文本类型：显示多行输入框，显示提示
      fieldValueDefault.style.display = 'none';
      fieldValueText.style.display = 'block';
      fieldValueDateWrapper.style.display = 'none';
      fieldValueHint.style.display = 'block';
      fieldValueHint.textContent = '✨ 字段值不限字数';
    }
  };
  
  // 监听字段类型选择变化
  fieldTypeSelect.addEventListener('change', updateFieldValueDisplay);
  
  // 初始状态显示灰色
  fieldTypeSelect.style.color = '#666';
  updateFieldValueDisplay();
  
  // 日期输入框点击事件：打开日期选择器
  fieldValueDate.addEventListener('click', () => {
    fieldValueDateHidden.showPicker();
  });
  
  // 隐藏的日期输入框变化时更新显示的文本
  fieldValueDateHidden.addEventListener('change', (e) => {
    if (e.target.value) {
      fieldValueDate.value = e.target.value;
      fieldValueDate.style.color = '#333';
    }
  });
  
  // 关闭对话框
  const closeDialog = () => {
    document.body.removeChild(dialog);
    document.head.removeChild(style);
  };
  
  closeBtn.onclick = closeDialog;
  cancelBtn.onclick = closeDialog;
  
  // 点击遮罩层关闭
  dialog.querySelector('.dialog-overlay').onclick = closeDialog;
  
  // 确认添加字段
  confirmBtn.onclick = async () => {
    const fieldName = fieldNameInput.value.trim();
    const fieldType = fieldTypeSelect.value;
    
    if (!fieldName) {
      alert('请输入字段名称');
      fieldNameInput.focus();
      return;
    }
    
    // 检查字段名称长度（最多8个字）
    if (fieldName.length > 8) {
      alert('字段名称不能超过8个字，请重新输入');
      fieldNameInput.focus();
      fieldNameInput.select();
      return;
    }
    
    // 检查是否选择了字段类型
    if (!fieldType) {
      alert('请选择字段类型');
      fieldTypeSelect.focus();
      return;
    }
    
    // 根据类型获取对应的值
    let fieldValue = '';
    if (fieldType === 'date') {
      fieldValue = fieldValueDateHidden.value || fieldValueDate.value;
    } else if (fieldType === 'text') {
      fieldValue = fieldValueText.value.trim();
    } else {
      // 默认情况下使用默认输入框的值
      fieldValue = fieldValueDefault.value.trim();
    }
    
    // 检查字段是否已存在
    if (moduleInfo.type === 'single') {
      if (config[moduleInfo.key] && config[moduleInfo.key].hasOwnProperty(fieldName)) {
        alert('该字段已存在，请使用其他名称');
        fieldNameInput.focus();
        fieldNameInput.select();
        return;
      }
      if (!config[moduleInfo.key]) {
        config[moduleInfo.key] = {};
      }
      config[moduleInfo.key][fieldName] = fieldValue;
      
      // 更新字段顺序
      if (!config.fieldOrder) {
        config.fieldOrder = {};
      }
      if (!config.fieldOrder[moduleInfo.key]) {
        config.fieldOrder[moduleInfo.key] = [];
      }
      if (!config.fieldOrder[moduleInfo.key].includes(fieldName)) {
        config.fieldOrder[moduleInfo.key].push(fieldName);
      }
      
      // 更新字段类型
      if (!config.fieldTypes) {
        config.fieldTypes = {};
      }
      if (!config.fieldTypes[moduleInfo.key]) {
        config.fieldTypes[moduleInfo.key] = {};
      }
      config.fieldTypes[moduleInfo.key][fieldName] = fieldType === 'date' ? 'date' : 'textarea';
    } else {
      const moduleKey = moduleInfo.key.replace('List', '');
      if (!config.fieldOrder) {
        config.fieldOrder = {};
      }
      if (!config.fieldOrder[moduleKey]) {
        config.fieldOrder[moduleKey] = [];
      }
      if (config.fieldOrder[moduleKey].includes(fieldName)) {
        alert('该字段已存在，请使用其他名称');
        fieldNameInput.focus();
        fieldNameInput.select();
        return;
      }
      
      if (!config[moduleInfo.key]) {
        config[moduleInfo.key] = [];
      }
      if (config[moduleInfo.key].length === 0) {
        config[moduleInfo.key].push({ id: `${moduleKey}_${Date.now()}` });
      }
      config[moduleInfo.key].forEach(item => {
        item[fieldName] = fieldValue;
      });
      config.fieldOrder[moduleKey].push(fieldName);
      
      // 更新字段类型
      if (!config.fieldTypes) {
        config.fieldTypes = {};
      }
      if (!config.fieldTypes[moduleKey]) {
        config.fieldTypes[moduleKey] = {};
      }
      config.fieldTypes[moduleKey][fieldName] = fieldType === 'date' ? 'date' : 'textarea';
    }
    
    await saveConfig();
    renderAllModules();
    closeDialog();
    
    // 滚动到新字段
    setTimeout(() => {
      const newField = document.querySelector(`[data-field-name="${fieldName}"][data-module-key="${moduleInfo.key}"]`);
      if (newField) {
        newField.scrollIntoView({ behavior: 'smooth', block: 'center' });
        // 高亮新字段
        newField.style.background = 'linear-gradient(135deg, #e8f5e9 0%, #c8e6c9 100%)';
        setTimeout(() => {
          newField.style.background = '';
        }, 1000);
      }
    }, 100);
    
    showMessage('字段已添加', 'success');
  };
  
  // 按Enter键确认
  fieldNameInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
      // 根据类型聚焦到对应的输入框
      if (fieldTypeSelect.value === 'date') {
        fieldValueDate.click(); // 点击打开日期选择器
      } else if (fieldTypeSelect.value === 'text') {
        fieldValueText.focus();
      } else {
        fieldValueDefault.focus();
      }
    }
  });
  
  // 聚焦到字段名称输入框
  setTimeout(() => fieldNameInput.focus(), 100);
}

// 填充输入框
async function fillInput(value) {
  if (!value || !value.trim()) {
    showMessage('字段值为空', 'error');
    return;
  }
  
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab) {
      showMessage('无法获取当前标签页', 'error');
      return;
    }
    
    chrome.tabs.sendMessage(tab.id, {
      action: 'fillInput',
      value: value.trim()
    }, (response) => {
      if (chrome.runtime.lastError) {
        showMessage('填充失败：' + chrome.runtime.lastError.message, 'error');
      } else if (response && response.success) {
        showMessage('填充成功', 'success');
      } else {
        showMessage('请先点击页面中的输入框', 'error');
      }
    });
  } catch (error) {
    console.error('填充失败:', error);
    showMessage('填充失败', 'error');
  }
}

// 检查当前活动的输入框
async function checkActiveInput() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab) return;
    
    chrome.tabs.sendMessage(tab.id, {
      action: 'getActiveInput'
    }, (response) => {
      if (chrome.runtime.lastError) {
        return;
      }
      
      if (response && response.hasInput && response.inputInfo) {
        updateTargetInput(response.inputInfo);
      } else {
        updateTargetInput(null);
      }
    });
  } catch (error) {
    // 忽略错误
  }
}

// 更新目标输入框信息
function updateTargetInput(inputInfo) {
  currentInputInfo = inputInfo;
  const targetField = document.getElementById('targetField');
  
  if (inputInfo) {
    let displayText = '';
    if (inputInfo.label) {
      displayText = inputInfo.label;
    } else if (inputInfo.placeholder) {
      displayText = inputInfo.placeholder;
    } else if (inputInfo.name) {
      displayText = inputInfo.name;
    } else if (inputInfo.id) {
      displayText = inputInfo.id;
    } else {
      displayText = '已选中输入框';
    }
    
    targetField.textContent = displayText;
    targetField.style.color = '#4caf50';
  } else {
    targetField.textContent = '请点击页面中的输入框';
    targetField.style.color = '#ff8c42';
  }
}

// 获取字段类型
function getFieldType(moduleKey, fieldName) {
  // 检查config中是否有fieldTypes配置
  if (config.fieldTypes && config.fieldTypes[moduleKey] && config.fieldTypes[moduleKey][fieldName]) {
    return config.fieldTypes[moduleKey][fieldName];
  }
  
  // 默认类型判断
  if (fieldName.includes('日期') || fieldName.includes('时间')) {
    return 'date';
  }
  if (fieldName.includes('描述') || fieldName.includes('职责') || fieldName.includes('内容') || fieldName.includes('成果') || fieldName.includes('情况') || fieldName.includes('评价')) {
    return 'textarea';
  }
  return 'input';
}

// 计算文本行数（与原插件保持一致）
function calculateTextLines(text, containerWidth, fontSize = 12, lineHeight = 1.4) {
  try {
    if (!text || !containerWidth || containerWidth <= 0) return 1;
    
    // 创建一个临时的测量元素
    const measureElement = document.createElement('div');
    measureElement.style.position = 'absolute';
    measureElement.style.visibility = 'hidden';
    measureElement.style.height = 'auto';
    measureElement.style.width = Math.max(containerWidth, 100) + 'px'; // 确保最小宽度
    measureElement.style.fontSize = fontSize + 'px';
    measureElement.style.lineHeight = lineHeight;
    measureElement.style.fontFamily = 'Arial, sans-serif';
    measureElement.style.wordWrap = 'break-word';
    measureElement.style.whiteSpace = 'pre-wrap';
    measureElement.style.padding = '6px 8px';
    measureElement.style.boxSizing = 'border-box';
    measureElement.style.left = '-9999px';
    measureElement.style.top = '-9999px';
    measureElement.textContent = text;
    
    document.body.appendChild(measureElement);
    
    const textHeight = measureElement.offsetHeight;
    const singleLineHeight = fontSize * lineHeight;
    const lines = Math.ceil(textHeight / singleLineHeight);
    
    document.body.removeChild(measureElement);
    
    return Math.max(1, lines);
  } catch (error) {
    console.error('计算文本行数时出错:', error);
    return 1; // 出错时返回1行
  }
}

// 根据文本行数调整按钮高度（与原插件保持一致）
function adjustButtonHeight(button, text, containerWidth) {
  try {
    if (!text || !containerWidth || !button) return;
    
    const lines = calculateTextLines(text, containerWidth, 12, 1.4);
    const baseHeight = 30; // 基础高度30px (8mm)
    const heightPerLine = 13; // 每行增加13px (约3.5mm)
    const additionalHeight = (lines - 1) * heightPerLine;
    const newHeight = baseHeight + additionalHeight;
    
    if (button && button.style) {
      button.style.height = newHeight + 'px';
      button.style.minHeight = newHeight + 'px';
    }
    
    console.log(`字段按钮高度调整: ${lines}行, 每行增加${heightPerLine}px (3.5mm), 新高度: ${newHeight}px`);
  } catch (error) {
    console.error('调整按钮高度时出错:', error);
  }
}

// 获取所有字段数据（用于一键填写）
function getAllFieldsData() {
  const fieldsData = {};
  
  modules.forEach(module => {
    const moduleData = config[module.key];
    if (!moduleData) return;
    
    if (module.type === 'single') {
      // 单组类型：直接获取字段
      if (typeof moduleData === 'object' && !Array.isArray(moduleData)) {
        Object.keys(moduleData).forEach(fieldName => {
          if (moduleData[fieldName] && moduleData[fieldName] !== '') {
            fieldsData[fieldName] = moduleData[fieldName];
          }
        });
      }
    } else {
      // 列表类型：获取第一项的字段
      if (Array.isArray(moduleData) && moduleData.length > 0) {
        const firstItem = moduleData[0];
        Object.keys(firstItem).forEach(fieldName => {
          if (fieldName !== 'id' && firstItem[fieldName] && firstItem[fieldName] !== '') {
            fieldsData[fieldName] = firstItem[fieldName];
          }
        });
      }
    }
  });
  
  return fieldsData;
}

// 一键填写
async function oneClickFill() {
  try {
    const fieldsData = getAllFieldsData();
    if (Object.keys(fieldsData).length === 0) {
      showMessage('没有可填充的字段数据', 'error');
      return;
    }
    
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab) {
      showMessage('无法获取当前标签页', 'error');
      return;
    }
    
    showMessage('正在自动匹配并填充...', 'info');
    
    chrome.tabs.sendMessage(tab.id, {
      action: 'oneClickFill',
      fieldsData: fieldsData
    }, (response) => {
      if (chrome.runtime.lastError) {
        showMessage('填充失败：' + chrome.runtime.lastError.message, 'error');
      } else if (response && response.success) {
        const filledCount = response.filledCount || 0;
        const totalCount = response.totalCount || 0;
        showMessage(`填充完成！成功填充 ${filledCount}/${totalCount} 个字段`, 'success');
      } else {
        showMessage('填充失败，请检查页面是否支持', 'error');
      }
    });
  } catch (error) {
    console.error('一键填写失败:', error);
    showMessage('一键填写失败', 'error');
  }
}

// 绑定事件
function bindEvents() {
  // 添加字段按钮（全局）
  const addFieldBtn = document.getElementById('addFieldBtn');
  if (addFieldBtn) {
    addFieldBtn.style.display = 'none'; // 隐藏全局添加按钮，使用模块内的添加按钮
  }
  
  // 一键填写按钮
  const oneClickFillBtn = document.getElementById('oneClickFillBtn');
  if (oneClickFillBtn) {
    oneClickFillBtn.onclick = oneClickFill;
  }
}

// 显示消息
function showMessage(message, type = 'info') {
  const messageEl = document.getElementById('message');
  if (!messageEl) return;
  
  messageEl.textContent = message;
  messageEl.className = `message ${type} show`;
  
  setTimeout(() => {
    messageEl.classList.remove('show');
  }, 2000);
}

// ==================== 兑换码验证功能 ====================

// 云函数URL配置
const CLOUD_FUNCTION_URL = 'https://xzwsxxb-0g3uev3r08d9ec03-1384540552.ap-shanghai.app.tcloudbase.com/redeem-code-verify';
const TOOL_ID = 'application-assistant'; // 网申提效助手的工具ID
const HEARTBEAT_TIMEOUT_MS = 60 * 1000; // 心跳超时时间：60秒

// 生成设备ID（基于浏览器指纹）
function getDeviceId() {
  let deviceId = localStorage.getItem('deviceId');
  if (!deviceId) {
    // 生成一个基于浏览器特征的设备ID
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    ctx.textBaseline = 'top';
    ctx.font = '14px Arial';
    ctx.fillText('Device fingerprint', 2, 2);
    const fingerprint = canvas.toDataURL();
    
    // 结合其他特征
    const features = [
      navigator.userAgent,
      navigator.language,
      screen.width + 'x' + screen.height,
      new Date().getTimezoneOffset(),
      fingerprint
    ].join('|');
    
    // 生成哈希
    let hash = 0;
    for (let i = 0; i < features.length; i++) {
      const char = features.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // Convert to 32bit integer
    }
    
    deviceId = 'device_' + Math.abs(hash).toString(36);
    localStorage.setItem('deviceId', deviceId);
  }
  return deviceId;
}

// 心跳机制
let heartbeatInterval = null;
let heartbeatBeforeUnloadHandler = null;

function startHeartbeat() {
  stopHeartbeat();
  const code = localStorage.getItem('redeemCode');
  const deviceId = getDeviceId();
  const sessionToken = localStorage.getItem('sessionToken');

  if (!code || !deviceId || !sessionToken) return;

  const sendHeartbeat = async () => {
    try {
      const response = await fetch(CLOUD_FUNCTION_URL, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          action: 'heartbeat',
          code,
          deviceId,
          sessionToken,
          toolId: TOOL_ID
        })
      });

      const result = await response.json().catch(() => null);
      if (!result || !result.success) {
        console.warn('心跳发送失败或会话无效:', result);
        // 如果心跳失败，可能需要重新验证
        if (result && (result.code === 'SESSION_MISMATCH' || result.code === 'SESSION_TIMEOUT')) {
          stopHeartbeat();
          showRedeemOverlay();
        }
      }
    } catch (error) {
      console.error('心跳发送失败:', error);
    }
  };

  // 先立即发送一次心跳
  sendHeartbeat();

  // 每 10 秒发送一次心跳
  heartbeatInterval = setInterval(sendHeartbeat, 10 * 1000);

  if (!heartbeatBeforeUnloadHandler) {
    heartbeatBeforeUnloadHandler = function() {
      if (heartbeatInterval) {
        clearInterval(heartbeatInterval);
      }

      const payload = JSON.stringify({
        action: 'unbindDevice',
        code,
        deviceId,
        sessionToken,
        toolId: TOOL_ID
      });

      if (navigator.sendBeacon) {
        const blob = new Blob([payload], { type: 'application/json' });
        navigator.sendBeacon(CLOUD_FUNCTION_URL, blob);
      }
    };

    window.addEventListener('beforeunload', heartbeatBeforeUnloadHandler);
  }
}

function stopHeartbeat() {
  if (heartbeatInterval) {
    clearInterval(heartbeatInterval);
    heartbeatInterval = null;
  }
  if (heartbeatBeforeUnloadHandler) {
    window.removeEventListener('beforeunload', heartbeatBeforeUnloadHandler);
    heartbeatBeforeUnloadHandler = null;
  }
}

// 验证兑换码
async function verifyRedeemCode(code) {
  try {
    const deviceId = getDeviceId();
    
    const response = await fetch(CLOUD_FUNCTION_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        action: 'verifyRedeemCode',
        code: code,
        deviceId: deviceId,
        toolId: TOOL_ID
      })
    });
    
    const result = await response.json();
    
    if (result.success) {
      if (result.sessionToken) {
        localStorage.setItem('sessionToken', result.sessionToken);
      } else {
        localStorage.removeItem('sessionToken');
      }
      // 验证成功，保存状态到localStorage
      localStorage.setItem('redeemCodeVerified', 'true');
      localStorage.setItem('redeemCode', code);
      localStorage.setItem('deviceId', deviceId);
      localStorage.setItem('redeemTime', Date.now().toString());
      return { success: true, message: result.message || '兑换码验证成功！' };
    } else {
      localStorage.removeItem('sessionToken');
      return { success: false, message: result.message || '兑换码验证失败，请检查兑换码是否正确' };
    }
  } catch (error) {
    console.error('兑换码验证错误:', error);
    return { 
      success: false, 
      message: '网络错误，请稍后重试。如果问题持续存在，请联系管理员。' 
    };
  }
}

// 使用云函数进行服务器端会话校验，防止仅依赖本地状态
async function checkPermissionWithServer() {
  const code = localStorage.getItem('redeemCode');
  const sessionToken = localStorage.getItem('sessionToken');
  const deviceId = getDeviceId();

  if (!code || !sessionToken || !deviceId) {
    return {
      success: false,
      authorized: false,
      code: 'MISSING_PARAMS',
      message: '缺少本地会话信息，需要重新输入兑换码'
    };
  }

  try {
    const response = await fetch(CLOUD_FUNCTION_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        action: 'checkPermission',
        code,
        deviceId,
        sessionToken,
        toolId: TOOL_ID
      })
    });

    const result = await response.json().catch(() => null);
    if (!result) {
      return {
        success: false,
        authorized: false,
        code: 'NETWORK_ERROR',
        message: '服务器返回数据异常，请重新输入兑换码'
      };
    }

    return {
      success: !!result.success && !!result.authorized,
      authorized: !!result.authorized,
      code: result.code || null,
      message: result.message || (result.authorized ? '授权有效' : '授权无效，请重新输入兑换码')
    };
  } catch (error) {
    console.error('checkPermission 调用失败:', error);
    // 出现网络错误时，为了安全起见，要求用户重新验证
    return {
      success: false,
      authorized: false,
      code: 'NETWORK_ERROR',
      message: '网络错误，请重新输入兑换码'
    };
  }
}

// 显示兑换码蒙版
function showRedeemOverlay() {
  const overlay = document.getElementById('redeemOverlay');
  if (overlay) {
    // 动态计算标题栏高度
    const dragHandle = document.getElementById('dragHandle');
    if (dragHandle) {
      const headerHeight = dragHandle.offsetHeight;
      overlay.style.top = headerHeight + 'px';
    }
    overlay.classList.remove('hidden');
    // 禁用所有功能按钮
    disableAllFeatures();
    // 禁用背景滚动
    document.body.style.overflow = 'hidden';
    const container = document.getElementById('popupContainer');
    if (container) {
      container.style.overflow = 'hidden';
    }
  }
}

// 隐藏兑换码蒙版
function hideRedeemOverlay() {
  const overlay = document.getElementById('redeemOverlay');
  if (overlay) {
    overlay.classList.add('hidden');
    // 启用所有功能按钮
    enableAllFeatures();
    // 恢复背景滚动
    document.body.style.overflow = '';
    const container = document.getElementById('popupContainer');
    if (container) {
      container.style.overflow = '';
    }
  }
}

// 禁用所有功能
function disableAllFeatures() {
  const fillButtons = document.querySelectorAll('.fill-btn, .field-fill-btn, #oneClickFillBtn, #addFieldBtn');
  fillButtons.forEach(btn => {
    btn.style.pointerEvents = 'none';
    btn.style.opacity = '0.5';
  });
}

// 启用所有功能
function enableAllFeatures() {
  const fillButtons = document.querySelectorAll('.fill-btn, .field-fill-btn, #oneClickFillBtn, #addFieldBtn');
  fillButtons.forEach(btn => {
    btn.style.pointerEvents = 'auto';
    btn.style.opacity = '1';
  });
}

// 显示兑换码消息
function showRedeemMessage(message, isSuccess) {
  const messageEl = document.getElementById('redeem-message');
  if (messageEl) {
    messageEl.textContent = message;
    messageEl.className = 'redeem-message ' + (isSuccess ? 'success' : 'error');
    messageEl.style.display = 'flex';
  }
}

// 初始化兑换码验证功能
function initRedeemCodeVerification() {
  // 检查是否已经验证过
  const isVerified = localStorage.getItem('redeemCodeVerified') === 'true';
  const redeemCode = localStorage.getItem('redeemCode');
  const sessionToken = localStorage.getItem('sessionToken');
  const redeemTime = parseInt(localStorage.getItem('redeemTime') || '0');
  
  // 先根据本地时间做一个粗略过期判断，避免非常旧的会话一直存在
  const now = Date.now();
  const isLocallyExpired = (now - redeemTime) > HEARTBEAT_TIMEOUT_MS * 3; // 放宽到约3个心跳周期

  if (isVerified && redeemCode && sessionToken && !isLocallyExpired) {
    // 本地认为已验证且未明显过期，进一步调用服务器进行权限校验
    (async () => {
      const checkResult = await checkPermissionWithServer();
      if (checkResult.success && checkResult.authorized) {
        // 服务器确认授权有效，隐藏蒙版并启动心跳
        hideRedeemOverlay();
        startHeartbeat();
      } else {
        console.warn('本地会话被服务器判定为无效，将重新要求输入兑换码:', checkResult);
        // 清理本地会话信息并强制用户重新输入兑换码
        localStorage.removeItem('redeemCodeVerified');
        localStorage.removeItem('redeemCode');
        localStorage.removeItem('sessionToken');
        localStorage.removeItem('redeemTime');
        showRedeemOverlay();
        if (checkResult.message) {
          showRedeemMessage(checkResult.message, false);
        }
      }
    })();
  } else {
    // 本地已明显过期或缺少信息，直接显示蒙版要求输入兑换码
    showRedeemOverlay();
  }
  
  // 绑定兑换码输入框和按钮事件
  const codeInput = document.getElementById('redeem-code-input');
  const codeBtn = document.getElementById('redeem-code-btn');
  const messageEl = document.getElementById('redeem-message');
  
  if (!codeInput || !codeBtn) {
    // 如果元素还未加载，延迟重试
    setTimeout(initRedeemCodeVerification, 100);
    return;
  }
  
  // 确保输入框可以交互
  codeInput.disabled = false;
  codeInput.readOnly = false;
  codeInput.style.pointerEvents = 'auto';
  codeInput.style.opacity = '1';
  
  // 确保按钮可以交互
  codeBtn.disabled = false;
  codeBtn.style.pointerEvents = 'auto';
  codeBtn.style.opacity = '1';
  
  const handleRedeemSubmit = async () => {
    if (codeBtn.disabled) {
      return;
    }
    
    const code = codeInput.value.trim();
    
    if (!code) {
      showRedeemMessage('请输入兑换码', false);
      return;
    }
    
    // 禁用按钮，显示加载状态
    codeBtn.disabled = true;
    codeBtn.textContent = '验证中...';
    if (messageEl) messageEl.textContent = '';
    
    try {
      const result = await verifyRedeemCode(code);

      if (result.success) {
        showRedeemMessage(result.message, true);
        // 延迟隐藏蒙版，让用户看到成功消息
        setTimeout(() => {
          hideRedeemOverlay();
        }, 1500);
        // 启动心跳
        startHeartbeat();
      } else {
        showRedeemMessage(result.message, false);
        codeBtn.disabled = false;
        codeBtn.textContent = '验证';
      }
    } catch (error) {
      console.error('验证过程出错:', error);
      showRedeemMessage('验证过程出错，请稍后重试', false);
      codeBtn.disabled = false;
      codeBtn.textContent = '验证';
    }
  };
  
  // 验证按钮点击事件
  codeBtn.addEventListener('click', handleRedeemSubmit, true);
  
  // 输入框回车事件
  codeInput.addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
      e.preventDefault();
      e.stopPropagation();
      handleRedeemSubmit();
    }
  }, true);
  
  // 确保输入框可以接收焦点
  codeInput.addEventListener('focus', function(e) {
    e.stopPropagation();
  }, true);
  
  // 确保输入框可以接收输入
  codeInput.addEventListener('input', function(e) {
    e.stopPropagation();
  }, true);
}

// 修改init函数，添加兑换码验证初始化
const originalInit = init;
init = async function() {
  // 先初始化兑换码验证
  initRedeemCodeVerification();
  
  // 然后执行原有的初始化
  await originalInit();
};

// 页面加载完成后初始化
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
